/*----------------------------------------------------------------------------
FILE:           c2b.c
DESCRIPTION:    Translates object code structure to EPROM bytes.
                Part of pub21k.c program.
AUTHOR:         Jim Donahue, Analog Devices DSP Division (617) 461-3672
DATE:           6/13/91
MODIFIED:	4/10/92 to fix shift constant to LONG in scanDR
----------------------------------------------------------------------------*/
#include "pub21k.h"

/*----------------------------------------------------------------------------
SUBROUTINE:     changeState
DESCRIPTION:    change state of Test Access Port
----------------------------------------------------------------------------*/
void changeState(char *str, BYTE_ARRAY *bytes, long *bitindex) 
{
#ifdef DEBUG
    printf ("changeState: bitindex = %d, byte = %X, index = %X\n", 
        *bitindex, *bitindex/4, *bitindex%4);
#endif
    while (*str != (char) NULL) {
        if (*str++ == '1') 
            SET_TMS(*bitindex,0);
        (*bitindex)++;
    }
}

/*----------------------------------------------------------------------------
SUBROUTINE:     scanIR
DESCRIPTION:    scan into Test Access Port Instruction register
----------------------------------------------------------------------------*/
void scanIR(char *str, BYTE_ARRAY *bytes, long *bitindex) 
{
#ifdef DEBUG
    printf ("scanIR: bitindex = %d, byte = %X, index = %X\n", 
        *bitindex, *bitindex/4, *bitindex%4);
#endif
    while (*str != (char) NULL) {
        if (*str++ == '1') 
            SET_TDI(*bitindex,0);
        (*bitindex)++;
    }

    (*bitindex)--;    /* must correct bitindex since exiting scan state     */
                    /*       (TMS=1) must coincide with sending last datum  */
}

/*----------------------------------------------------------------------------
SUBROUTINE:     scanDR
DESCRIPTION:    scan into Test Access Port Data register
----------------------------------------------------------------------------*/
void scanDR(int num, char strobeState, OBJCODE *code, 
            BYTE_ARRAY *bytes, long *bitindex) 
{
    int     i;

#ifdef DEBUG
    printf ("scanDR: bitindex = %d, byte = %X, index = %X", 
        *bitindex, *bitindex/4, *bitindex%4);
    if (strobeState) printf (", active strobes\n");
        else printf (", inactive strobes\n");
#endif

    /* Set background mask of all ones - this assures inactive (high)       */
    /*      memory strobes and interrupts.                                  */
    /* All further operations will be selective clearing of TDI bits        */
    for (i=0; i<DRLOCS; i++) 
        SET_TDI(*bitindex,i);

    /* RESET, FLAG0-3 enables always active low */
    CLR_TDI(*bitindex,RESET);
    CLR_TDI(*bitindex,FLG0OE);
    CLR_TDI(*bitindex,FLG1OE);
    CLR_TDI(*bitindex,FLG2OE);
    CLR_TDI(*bitindex,FLG3OE);

    if (strobeState==ACTIVE) {
        /* Activate bank selects and write lines */
        if ((code+num)->bank==BANK0) 
            CLR_TDI(*bitindex,PMS0);
        else 
            CLR_TDI(*bitindex,PMS1);
        CLR_TDI(*bitindex,PMWR);
    }

    for (i=0;i<24;i++)
        if (!((code+num)->pma & (1L<<i))) 
            CLR_TDI(*bitindex,PMA(i));

    for (i=0;i<32;i++)
        if (!((code+num)->pmdl & (1L<<i))) 
            CLR_TDI(*bitindex,PMD(i));

    for (i=32;i<48;i++)
        if (!((code+num)->pmdu & (1L<<(i-32)))) 
            CLR_TDI(*bitindex,PMD(i));

    /* DMD[15:0] used as a flag to indicate which instruction   */
    /*  is being scanned down                                   */
    for (i=0;i<8;i++)
        if (!(num & (1L<<i))) 
            CLR_TDI(*bitindex,DMD(i));

    /* must correct bitindex since exiting scan state (TMS=1)   */
    /*      must coincide with sending last datum               */
    *bitindex = *bitindex + DRLOCS - 1;
}

/*----------------------------------------------------------------------------
SUBROUTINE:     initializeTAP
DESCRIPTION:    initialize Test Access Port
----------------------------------------------------------------------------*/
void initializeTAP(OBJCODE *code, BYTE_ARRAY *bytes, long *bitindex) 
{

    /* reset TAP, enter RT/IDLE state */
    changeState(TLRESET, bytes, bitindex);
    changeState(RST2RTI, bytes, bitindex);

    /* enter IRSHIFT state, scan in SAMPLE/PRELOAD instruction */
    changeState(RTI2IRS, bytes, bitindex);
    scanIR(PRELD, bytes, bitindex);
    changeState(S2RTI, bytes, bitindex);

    /* enter DRSHIFT state, scan in RESET active, mem. strobes inactive */
    changeState(RTI2DRS, bytes, bitindex);
    scanDR(0, INACTIVE, code, bytes, bitindex);
    changeState(S2RTI, bytes, bitindex);

    /* enter IRSHIFT state, scan in INTEST instruction */
    changeState(RTI2IRS, bytes, bitindex);
    scanIR(INTEST, bytes, bitindex);
    changeState(S2RTI, bytes, bitindex);

}

/*----------------------------------------------------------------------------
SUBROUTINE:     scanCode
DESCRIPTION:    scan in instructions
----------------------------------------------------------------------------*/
void scanCode(int num, OBJCODE *code, BYTE_ARRAY *bytes, long *bitindex) 
{
#ifdef DEBUG
    printf ("scanCode: processing code line %d\n", num);
#endif

    /* enter DRSHIFT state, scan in instruction with inactive strobes       */
    changeState(RTI2DRS, bytes, bitindex);
    scanDR(num, INACTIVE, code, bytes, bitindex);
    changeState(S2RTI, bytes, bitindex);

    /* enter DRSHIFT state, scan in instruction with ACTIVE strobes         */
    changeState(RTI2DRS, bytes, bitindex);
    scanDR(num, ACTIVE, code, bytes, bitindex);
    changeState(S2RTI, bytes, bitindex);

}

/*----------------------------------------------------------------------------
SUBROUTINE:     terminateTAP
DESCRIPTION:    terminate Test Access Port
----------------------------------------------------------------------------*/
void terminateTAP(int num, OBJCODE *code, BYTE_ARRAY *bytes, long *bitindex) 
{
    /* enter DRSHIFT state, scan in last instruction with inactive strobes  */
    changeState(RTI2DRS, bytes, bitindex);
    scanDR(num, INACTIVE, code, bytes, bitindex);
    changeState(S2RTI, bytes, bitindex);

    /* fill remainder of EPROM with 1's in TMS bits. This will  */
    /*      hold TAP in TLRESET                                 */
#ifdef DEBUG
    printf ("terminateTAP: setting remaining TMS bits\n");
    printf ("terminateTAP: bitindex = %d, byte = %X, index = %X\n", 
        *bitindex, *bitindex/4, *bitindex%4);
#endif
    while (*bitindex < (EPROMSZ*4L)) {
            SET_TMS (*bitindex, 0);
            (*bitindex)++;      
    }
}

/*----------------------------------------------------------------------------
SUBROUTINE:     code2bytes
DESCRIPTION:    translates code structure to EPROM bytes
----------------------------------------------------------------------------*/
void code2bytes(OBJCODE *code, BYTE_ARRAY *bytes)
{
    int     i;
    long    bitindex = 0L;

    /* initialize the Test Access Port      */
    initializeTAP(code, bytes, &bitindex);

    /* process all valid entries in code structure */
    for (i=0; i<MAXLINES && code[i].bank!=ENDFLG; i++) {
        scanCode(i, code, bytes, &bitindex);
    }

    /* we are outta here!                   */
    terminateTAP(i-1, code, bytes, &bitindex);  

}
