#include <stdio.h>
#include <stddef.h>
#include <malloc.h>
#include <string.h>

/*----------------------------------------------------------------------------
Constant Definitions
----------------------------------------------------------------------------*/
#define EPROMSZ     32768L      /* EPROM Size in bytes                      */
#define LINELEN     256         /* Maximum Line Length                      */
#define MAXLINES    256         /* Maximum lines of code                    */
#define ARGCHARS    33          /* Maximum characters in an argument        */

/* bank selects in code structure */
#define BANK0       0           /* Bank 0 == PMS0 active                    */
#define BANK1       1           /* Bank 1 == PMS0 active                    */
#define ENDFLG      -1          /* flag indicating end of code              */

/* strobe states for scanDR */
#define INACTIVE    0
#define ACTIVE      1

/*----------------------------------------------------------------------------
Scan Path Definitions
----------------------------------------------------------------------------*/
/* define TMS values to sequence to next state */
#define TLRESET     "11111111"      /* from any state to TLRESET state      */
#define RST2RTI     "0"             /* from TLRESET to RT/IDLE              */
#define RTI2DRS     "100"           /* from RT/IDLE to DRSHIFT              */
#define RTI2IRS     "1100"          /* from RT/IDLE to DRSHIFT              */
#define S2RTI       "110"           /* from DRSHIFT or IRSHIFT to RT/IDLE   */

/* define TDI values for Instruction Register                               */
/*  Note: these appear bit-reversed from the user's guide since the         */
/*        user's guide shows them with the lsb on the LEFT                  */
#define PRELD       "0001"
#define INTEST      "0011"

/* mapping of 21020's pins to scan path */
#define DRLOCS          286
#define RESET           (DRLOCS - 7)
#define PMWR            (DRLOCS - 11)
#define PMA(offset)     (DRLOCS - (257+offset))
#define PMD(offset)     (DRLOCS - (109 - 2*offset))
#define PMS0            (DRLOCS - 283)
#define PMS1            (DRLOCS - 282)
#define FLG3            (DRLOCS - 235)
#define FLG2            (DRLOCS - 237)
#define FLG1            (DRLOCS - 239)
#define FLG0            (DRLOCS - 241)
#define FLG3OE          (DRLOCS - 228)
#define FLG2OE          (DRLOCS - 233)
#define FLG1OE          (DRLOCS - 242)
#define FLG0OE          (DRLOCS - 248)
#define DMD(offset)     (DRLOCS - (112 + 2*offset))

/*----------------------------------------------------------------------------
Macros for Bit Manipulation in EPROM bytes array
----------------------------------------------------------------------------*/
#define SET_TDI(index,offset) \
    bytes[(index+offset)/4] |= (1 << ((index+offset)%4))

#define SET_TMS(index,offset) \
    bytes[(index+offset)/4] |= (16 << ((index+offset)%4))

#define CLR_TDI(index,offset) \
    bytes[(index+offset)/4] &= ~(1 << ((index+offset)%4))

#define CLR_TMS(index,offset) \
    bytes[(index+offset)/4] &= ~(16 << ((index+offset)%4))


/*----------------------------------------------------------------------------
Typedefs
----------------------------------------------------------------------------*/
typedef struct objcode {
    char bank;              /* structure describing object code to be       */
    unsigned long pma;      /*      scanned into Program Memory             */
    unsigned long pmdu;
    unsigned long pmdl;
} OBJCODE;

typedef struct header_t {
    unsigned char width;
    unsigned char version;
    unsigned char flags;
    unsigned char uflags;
    unsigned long address;
    unsigned long length;
} HEADER;

typedef unsigned long ULONG;

#ifdef MSDOS

typedef unsigned char huge BYTE_ARRAY;
#define CALLOC halloc    

#else

typedef unsigned char BYTE_ARRAY;
#define CALLOC calloc

#endif

/*----------------------------------------------------------------------------
Function prototypes
----------------------------------------------------------------------------*/
int getData(FILE *ip, HEADER *hdr, ULONG bankaddr, OBJCODE *code);
int getHeader(FILE *ip, HEADER *hdr);
void spl2code(FILE *ip, ULONG bankaddr, OBJCODE *code);
void changeState(char *str, BYTE_ARRAY *bytes, long *bitindex);
void scanIR(char *str, BYTE_ARRAY *bytes, long *bitindex);
void scanDR(int num, char strobeState, OBJCODE *code,
            BYTE_ARRAY *bytes, long *bitindex);
void initializeTAP(OBJCODE *code, BYTE_ARRAY *bytes, long *bitindex);
void scanCode(int num, OBJCODE *code, BYTE_ARRAY *bytes, long *bitindex);
void terminateTAP(int num, OBJCODE *code, BYTE_ARRAY *bytes, long *bitindex);
void code2bytes(OBJCODE *code, BYTE_ARRAY *bytes);
void makeSrecord(FILE *outfilep, long num, BYTE_ARRAY *bytes);
void makeStail(FILE *outfilep);
void bytes2burn(BYTE_ARRAY *bytes, FILE *outfilep);
ULONG stox (char *str, int chars);
