/*----------------------------------------------------------------------------
FILE:           s2c.c
DESCRIPTION:    Reads splitter stacked format file, creates code structure.
                Part of pub21k.c program.
AUTHOR:         Jim Donahue, Analog Devices DSP Division (617) 461-3672
DATE:           6/13/91
----------------------------------------------------------------------------*/
#include "pub21k.h"
#define PMBYTES     6

/*----------------------------------------------------------------------------
SUBROUTINE:     getData
DESCRIPTION:    gets Data from a splitter stacked format file

        The format of each line is simply 12 characters "UUUULLLLLLLL",
        where "UUUU" is the 4-character (2 bytes) upper bits of the
        instruction, and "LLLLLLLL" is the 8-character lower bits.
----------------------------------------------------------------------------*/
int getData(FILE *ip, HEADER *hdr, ULONG bankaddr, OBJCODE *code)
{
    int             i;
    char            upper[5];
    char            lower[9];
    char            line[LINELEN];
    unsigned long   addr;
    int             max;
    static int      codeindex = 0;

    addr = hdr->address;
    max = hdr->length/PMBYTES;
    for (i=0; i<max; i++, addr++, codeindex++) {
        if (fgets(line,LINELEN,ip)==NULL) {
            fprintf (stderr, "ERROR: Premature EOF in input file\n");
            exit(1);
        }
        if (sscanf(line, "%4s%8s", upper, lower) != 2) return (-1); 
        if (addr < bankaddr) {
            (code+codeindex)->bank = BANK0;         
        } else {
            (code+codeindex)->bank = BANK1;
        }
        (code+codeindex)->pma = addr;
        (code+codeindex)->pmdl = stox(lower,9);
        (code+codeindex)->pmdu = stox(upper,5);
    }
    return(0);
}

/*----------------------------------------------------------------------------
SUBROUTINE:     getHeader
DESCRIPTION:    gets a splitter stacked format header

        The header format is "WWVVFFUUAAAAAAAALLLLLLLL", where
        WW = width of address & length fields   (2 characters)
        VV = version #                          (2 characters)
        FF = splitter flags                     (2 characters)
        UU = user flags                         (2 characters)
        AAAAAAAA = address                      (8 characters)
        LLLLLLLL = length                       (8 characters)

----------------------------------------------------------------------------*/
int getHeader(FILE *ip, HEADER *hdr)
{
    char            line[LINELEN], *lp;

    lp = line;
    if (fgets(line,LINELEN,ip)==NULL) return (-1);

    /* build header structure, although only address and length are used */
    hdr->width      = (char) stox(lp, 2); lp+=2;
    hdr->version    = (char) stox(lp, 2); lp+=2;
    hdr->flags      = (char) stox(lp, 2); lp+=2;
    hdr->uflags     = (char) stox(lp, 2); lp+=2;
    hdr->address    =        stox(lp, hdr->width/4); lp+=hdr->width/4;
    hdr->length     =        stox(lp, hdr->width/4);

    return(0);
}

/*----------------------------------------------------------------------------
SUBROUTINE:     spl2code
DESCRIPTION:    read splitter stacked format file, generate code structure
----------------------------------------------------------------------------*/
void spl2code(FILE *ip, ULONG bankaddr, OBJCODE *code)
{
    HEADER hdr;

    /* read headers until they're gone, put data into code structure */
    while (!getHeader(ip,&hdr)) {
#ifdef DEBUG
    printf ("spl2code: header addr=%08.8X, length(bytes) = %08.8X\n", 
            hdr.address, hdr.length);
#endif
        /* get data following current header */
        if (getData(ip, &hdr, bankaddr, code)) {
            fprintf (stderr, "ERROR: data format in input file\n", ip);
            exit(1);
        }
        
    }
}

